/*

Copyright (c) 2004 Rusty Russell, IBM Corporation

This file is part of nfsim.

nfsim is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

nfsim is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with nfsim; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <core.h>
#include <tui.h>
#include "gen_ip.h"

/* Simple code to generate and send ICMP errors packet.  */
static bool
create_icmp(struct packet *packet, const struct packet *orig, int origlen,
	    const char *srcip, const char *typestr, const char *codestr)
{
	const struct in_addr *addr;
	int type, code;
	/* Length = IP header + ICMP header + Orig IP header length + 8 */
	u_int16_t len = sizeof(struct iphdr) + sizeof(struct icmphdr)
		+ origlen;

	if (origlen > htons(orig->iph.tot_len)) {
		nfsim_log(LOG_UI, "gen_err len %i greater than packet %i",
			  origlen, htons(orig->iph.tot_len));
		return false;
	}

	addr = dotted_to_addr(srcip);
	if (!addr) {
		nfsim_log(LOG_UI, "Bad src address `%s'", srcip);
		return false;
	}

	packet->pad = 0;
	memset(&packet->ehdr, 0, sizeof(packet->ehdr));
	packet->ehdr.h_proto = htons(ETH_P_IP);
	packet->iph = ((struct iphdr)
		      {
#if __BYTE_ORDER == __LITTLE_ENDIAN
			      5, 4,
#else
			      4, 5,
#endif
			      0, htons(len), 0, 0, 255, IPPROTO_ICMP, 0,
			      addr->s_addr, orig->iph.saddr });
	packet->iph.check = ~csum_partial(&packet->iph, sizeof(packet->iph), 0);

	if ((type = string_to_number(typestr, 0, 255)) < 0) {
		nfsim_log(LOG_UI, "Bad icmp type: `%s'", typestr);
		return false;
	}

	if ((code = string_to_number(codestr, 0, 255)) < 0) {
		nfsim_log(LOG_UI, "Bad icmp code: `%s'", codestr);
		return false;
	}

	packet->u.icmph.type = type;
	packet->u.icmph.code = code;
	packet->u.icmph.checksum = 0;
	packet->u.icmph.un.gateway = 0; /* MBZ */

	memcpy(&packet->u.icmph + 1, &orig->iph, origlen);

	packet->u.icmph.checksum
		= ~csum_partial(&packet->u,
				ntohs(packet->iph.tot_len)
				- sizeof(struct iphdr),
				0);
	return true;
}

static void gen_err_help(int argc, char **argv)
{
#include "gen_err-help:gen_err"
/*** XML Help:
    <section id="c:gen_err">
     <title><command>gen_err</command></title>
     <para>Generate an ICMP error packet and send it through the protocol
      layers</para>
     <cmdsynopsis>
      <command>gen_err</command>
      <arg choice="opt">IF=<replaceable>interface</replaceable></arg>
      <arg choice="opt">LEN=<replaceable>length</replaceable></arg>
      <arg choice="req">srcip</arg>
      <arg choice="req">icmp-type</arg>
      <arg choice="req">icmp-code</arg>
      <arg choice="req"><replaceable>gen-ip-options</replaceable></arg>
     </cmdsynopsis>

     <para><command>gen_err</command> will use
      <command>gen_ip</command> to construct an IP packet, generate an
      ICMP error for that packet, as would be generated by a host at
      <arg>srcip</arg>, then send it on the specified interface.  If
      no interface is specified, the packet will be sent from as if
      from the local host.  The <arg>length</arg> is the number of
      bytes to include from the original packet, default 28. </para>

     <para>See <command>gen_ip</command> for how to specify the
     options for the initial packet.</para> </section>
*/
}

static bool gen_err(int argc, char *argv[])
{
	struct packet orig, packet;
	int len = 28;
	char *interface = NULL;
	char *dump_flags = NULL;

	if (argc > 1 && strncmp(argv[1], "IF=", 3) == 0) {
		interface = argv[1] + 3;
		argc--;
		argv++;
	}

	if (argc > 1 && strncmp(argv[1], "LEN=", 4) == 0) {
		len = string_to_number(argv[1] + 4, 0, 65535);
		argc--;
		argv++;
	}

	if (!parse_packet(&orig, argc-3, argv+3, &dump_flags)) {
		gen_err_help(0, NULL);
		return false;
	}

	if (!create_icmp(&packet, &orig, len, argv[1], argv[2], argv[3]))
		return false;

	return send_packet(&packet, interface, dump_flags);
}

static void init(void)
{
	tui_register_command("gen_err", gen_err, gen_err_help);
}

init_call(init);
