
#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <libspe2.h>
#include <pthread.h>

#include "png.h"
#include "fractal.h"
#include "parse-fractal.h"

#define DEFAULT_PARAMSFILE "fractal.data"
#define DEFAULT_OUTFILE "fractal.png"

extern spe_program_handle_t spe_fractal;

struct spe_thread {
	spe_context_ptr_t ctx;
	pthread_t pthread;
	struct spe_args args __attribute__((aligned(SPE_ALIGN)));
};

void *spethread_fn(void *data)
{
	struct spe_thread *spethread = data;
	uint32_t entry = SPE_DEFAULT_ENTRY;

	/* run the context, passing the address of our args structure to
	 * the 'argv' argument to main() */
	spe_context_run(spethread->ctx, &entry, 0,
			&spethread->args, NULL, NULL);

	return NULL;
}

int main(int argc, char **argv)
{
	struct spe_thread thread;
	struct fractal_params *fractal;
	const char *outfile, *paramsfile;
	int opt;

	/* set up default arguments */
	paramsfile = DEFAULT_PARAMSFILE;
	outfile = DEFAULT_OUTFILE;

	/* parse arguments into datafile and outfile  */
	while ((opt = getopt(argc, argv, "p:o:")) != -1) {
		switch (opt) {
		case 'p':
			paramsfile = optarg;
			break;
		case 'o':
			outfile = optarg;
			break;
		default:
			fprintf(stderr, "Usage: %s [-p paramsfile] "
						"[-o outfile]\n", argv[0]);
			return EXIT_FAILURE;
		}
	}

	/* parse the input datafile */
	fractal = parse_fractal(paramsfile);
	if (!fractal)
		return EXIT_FAILURE;

	/* allocate our image buffer */
	fractal->imgbuf = memalign(SPE_ALIGN, sizeof(*fractal->imgbuf)
			* fractal->rows * fractal->cols);

	if (!fractal->imgbuf) {
		perror("memalign");
		return EXIT_FAILURE;
	}

	memcpy(&thread.args.fractal, fractal, sizeof(*fractal));

	thread.ctx = spe_context_create(0, NULL);
	spe_program_load(thread.ctx, &spe_fractal);
	pthread_create(&thread.pthread, NULL, spethread_fn, &thread);

	pthread_join(thread.pthread, NULL);

	/* compress and write to outfile */
	if (write_png(outfile, fractal->rows, fractal->cols, fractal->imgbuf))
		return EXIT_FAILURE;

	return EXIT_SUCCESS;
}
