/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 *
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file scheduler.h
 *
 * Allows delayed execution of tasks.
 *
 * Three types of task are defined - KEEPALIVE, KEEPALIVE_TIMEOUT and REMOVAL
 * tasks.
 *
 * A keepalive task causes a NECP_KEEPALIVE packet to be sent to a server. The
 * pointer to the server to send to is in the data1 field of the task. When a
 * keepalive task is processed, a keepalive timeout task is automatically
 * scheduled.
 *
 * A keepalive timeout is scheduled when a keepalive is sent. If a timeout task
 * is executed before the cancel_keepalive() method is called, the number of
 * retries for a server is incremented. When the number of retries reaches
 * NECP_RETRIES (defined in necp.h), the server is considered dead.
 *
 * A removal task is scheduled for dead or stopped services. There are two
 * types of removal tasks: a server-wide task and a single service task. A
 * single server task checks one service, and a server wide task checks all
 * tasks for a server. When a service's active connections (obtained from the
 * ipvs_interface object) reaches 0, the service is removed from the tables.
 *
 * @todo Scheduler granularity is currently one second - change this to ms ?
 */

#ifndef __HAVE_SCHEDULER_H
#define __HAVE_SCHEDULER_H 1

#include "feedbackd-master.h"
#include "linkedlist.h"

/**
 * Types of task
 */
typedef enum {
	TASK_OP_KEEPALIVE		= 0x01,	/**< Send a keepalive */
	TASK_OP_KEEPALIVE_TIMEOUT	= 0x02,	/**< keepalive timeout */
	TASK_OP_REMOVAL			= 0x04, /**< Check the number of active
					             connections to a server */
} task_op;


/**
 * Task object
 */
typedef struct {
	time_t		time;		/**< Time (seconds since epoch) that
					     the task is to be executed */
	task_op		operation;	/**< Task operation */
	unsigned int	data1;		/**< Task-specific data field */
	unsigned int	data2;		/**< Task specific data field */
} task_t;

/**
 * The current (ordered!) list of tasks
 */
list tasks;

/**
 * Schedules a keepalive request for a server. This also schedules the
 * keepalive timeout
 *
 * @param server The server to send the keepalive request to
 */
int schedule_keepalive(struct server *server);

/**
 * Schedules the removal of a service
 * @param service The service to remove when activeconns = 0
 */
int schedule_removal(struct service *service);

/**
 * Process the task list. This function is called when get_timeout reaches 0
 */
void process_tasklist(void);

/**
 * Returns the time of the next task in the task list. This should be used to
 * set the time of the next process_tasklist() call.
 *
 * @return the time of the next task, or 0 if there is no next task
 */
int get_timeout(void);

/**
 * Unschedule a server - this removes keepalive and keepalive timeout tasks
 * from the tasklist - should be called when the connection to a server is
 * closed
 *
 * @param server the server to cancel tasks for
 * @return Zero if the server was unscheduled, non-zero if no tasks were removed
 */
int unschedule_server(struct server *server);

/**
 * Get the next task to execute, and leave it in the tasklist. If there is no
 * next task, NULL is returned.
 *
 * @return a pointer to the next task to execute.
 */
task_t *tasklist_peek(void);

/**
 * Get the next task to execute, removing it from the list. The task must be
 * free()ed once removed.
 *
 * @return a pointer to the next task to execute
 */
task_t *tasklist_pop(void);

/**
 * Place a new task into the task list. Tasks should be created with malloc()
 * or friends.
 *
 * @param task the task to schedule
 */
void tasklist_add(task_t *task);

/**
 * Cancel a keepalive timeout scheduled for a server. This is typically called
 * when a keepalive is received from the server
 *
 * @param server the server for the keepalive timeout task
 * @return Zero if the cancel was sucessful, non-zero otherwise
 */
int cancel_keepalive_timeout(struct server *server);

/**
 * Prints the tasklist. If NDEBUG is defined, this function does nothing
 */
void print_tasklist(void);

/**
 * Cancels a removal task - this should be called if a NECP_START is received
 * for a server that is due to be removed
 * @param service the service to cancel the removal task
 */
int cancel_removal(struct service *service);

/**
 * Function to compare two tasks, to allow ordered inserts into the tasklist.
 * @param task1 A task to compare
 * @param task2 A task to compare
 * @return 0 if x happens after y, 1 otherwise
 */
int comp_tasks(void *task1, void *task2);

/**
 * Check is a server is referenced by any services
 * @param server the server to check for services
 * @return the number of  services that reference this server
 */
int server_inuse(struct server *server);


/**
 * Set the interval for task occurence
 * @param type The type of the task:
 *  <ul>
 *   <li>TASK_OP_KEEPALIVE for the interval between keepalive requests</li>
 *   <li>TASK_OP_KEEPALIVE_TIMEOUT for the time elapsed before a keepalive is
 *    considered lost</li>
 *   <li>TASK_OP_REMOVAL for the time between removal checks for a service</li>
 *  </ul>
 * @param interval The time (in seconds) to assign to the interval
 */
void set_interval(task_op type, int interval);

void scheduler_set_keepalive_timeout(int x);
void scheduler_set_keepalive_interval(int x);
void scheduler_set_keepalive_retries(int x);
void scheduler_set_removal_interval(int x);

#endif /* __HAVE_SCHEDULER_H */
