/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file scheduler.c
 */
#include <time.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <assert.h>
#include <errno.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include "necp_handler.h"
#include "necp_listener.h"
#include "log.h"
#include "scheduler.h"
#include "necp.h"
#include "ipvs_interface.h"

/**
 * Timeout (in seconds) before a NECP_KEEPALIVE request is considered to be
 * lost
 */
static int keepalive_timeout = 3;

/**
 * Time (in seconds) between NECP_KEEPALIVE requests being sent to each server
 */
static int keepalive_interval = 10;

/**
 * Number of NECP_KEEPALIVES to retry before we consider a server dead
 */
static int keepalive_retries = NECP_RETRIES;

/**
 * Time (in seconds) between checks of the number of active connections, which
 * must be 0 before a server is removed from the table.
 */
static int removal_interval = 10;


void scheduler_set_keepalive_timeout(int x)
{
	keepalive_timeout = x;
}

void scheduler_set_keepalive_interval(int x)
{
	keepalive_interval = x;
}

void scheduler_set_keepalive_retries(int x)
{
	keepalive_retries = x;
}

void scheduler_set_removal_interval(int x)
{
	removal_interval = x;
}

int schedule_keepalive(struct server *server)
{
	task_t *task;
	listitem *i;

	log_printf(LOG_VDEBUG, "Scheduling KEEPALIVE for server %s",
	                server->name);

	/* make sure there are no pending keepalives for this same server */
	list_for_each(tasks, i) {
		task = (task_t *)(i->item);
		if (task->operation == TASK_OP_KEEPALIVE &&
		    task->data1 == (int)server) {
			return 0;
			log_printf(LOG_VDEBUG,
					"Keepalive task already present.");
		}
	}


	if (!(task = malloc(sizeof(task_t)))) {
		perror("malloc");
		return -1;
	}

	/* create a keepalive task for KEEPALIVE_INTERVAL seconds
	 * in the future */
	task->time = time(NULL) + keepalive_interval;
	task->operation = TASK_OP_KEEPALIVE;
	task->data1 = (int) server;

	list_add_ordered(&tasks, task, comp_tasks);
	log_printf(LOG_VDEBUG, "Task list now has %d items", tasks.count);

	return 0;
}

int schedule_removal(struct service *service)
{
	task_t *task;
	listitem *i;

	log_printf(LOG_VDEBUG, "Scheduling removal for service (server %s, "
			"port %d)", service->server->name,
			ntohs(service->port));

	/* make sure there are no pending removals for this same service */
	list_for_each(tasks, i) {
		task = (task_t *)(i->item);
		if (task->operation == TASK_OP_REMOVAL &&
		    task->data1 == (int)service) {
			log_printf(LOG_VDEBUG, "Removal task already present.");
			return 0;
		}
	}


	if (!(task = malloc(sizeof(task_t)))) {
		perror("malloc");
		return -1;
	}

	/* create a removal task for REMOVAL_INTERVAL seconds in
	 * the future */
	task->time = time(NULL) + removal_interval;
	task->operation = TASK_OP_REMOVAL;
	task->data1 = (int) service;

	list_add_ordered(&tasks, task, comp_tasks);
	log_printf(LOG_VDEBUG, "Task list now has %d items", tasks.count);

	return 0;
}

int cancel_keepalive_timeout(struct server *server)
{
	task_t *task;
	listitem **i;

	i = &(tasks.head);

	/* step through the linked list of tasks */
	while (*i) {
		task = (task_t *)(*i)->item;
		if (task->operation == TASK_OP_KEEPALIVE_TIMEOUT &&
		    task->data1 == (int) server) {
			/* remove this matching task */
			listitem *tmp = *i;
			*i = (*i)->next;
			free(tmp);
			free(task);
			tasks.count--;
			return 0;
		}
		/* step to next task */
		i = &((*i)->next);
	}

	return -1;

}


int cancel_removal(struct service *service)
{
	task_t *task;
	listitem **i;

	i = &(tasks.head);

	/* step through the linked list of tasks */
	while (*i) {
		task = (task_t *)(*i)->item;
		if (task->operation == TASK_OP_REMOVAL &&
		    task->data1 == (int) service) {
			/* remove this matching task */
			listitem *tmp = *i;
			*i = (*i)->next;
			free(tmp);
			free(task);
			tasks.count--;
			return 0;
		}
		/* step to the next task */
		i = &((*i)->next);
	}

	return -1;

}

void process_tasklist()
{
	task_t *task, *new_task;
	listitem *tmp;
	struct server *server = NULL;
	struct service *service;

	log_printf(LOG_VDEBUG, "Processing tasklist");
	/* step through tasks that have reached their scheduled time. since
	 * the list is ordered, we can stop at the first task that is sceduled
	 * after time(NULL) */
	while (tasks.head) {
		task = (task_t *)tasks.head->item;
		if (task->time > time(NULL))
			break;

		/* take the current task off the list now, so that new tasks
		 * added don't need to skip past this one. */
		tmp = tasks.head;
		tasks.head = tasks.head->next;
		tasks.count--;
		free(tmp);
		log_printf(LOG_VDEBUG, "Task list now has %d items",
				tasks.count);

		switch (task->operation) {

		case TASK_OP_KEEPALIVE: /* a keepalive event */

			log_printf(LOG_DEBUG, "Keepalive task raised");

			/* send the keepalive packet */
			send_keepalive((struct server *)task->data1);

			/* create & schedule the timeout for this keepalive
			 * packet */
			if (!(new_task = malloc(sizeof(task_t)))) {
				perror("malloc");
				break;
			}

			new_task->time = time(NULL) + keepalive_timeout;
			new_task->operation = TASK_OP_KEEPALIVE_TIMEOUT;
			new_task->data1 = task->data1;

			log_printf(LOG_VDEBUG, "Adding a Keepalive timeout for "
					"server %s",
					((struct server *)(task->data1))->name);
			list_add_ordered(&tasks, new_task, comp_tasks);
			log_printf(LOG_VDEBUG, "Task list now has %d items",
			    tasks.count);

			break;

		case TASK_OP_KEEPALIVE_TIMEOUT: /* timeout for a keepalive */

			log_printf(LOG_DEBUG, "Keepalive timeout task raised");
			server = (struct server *) task->data1;

			/* if this timeout is the last, remove close the
			 * connection to the server */
			if (++(server->retries) >= keepalive_retries) {
				log_printf(LOG_WARN,
				    "Server %s has timed out %d times, "
				    "removing",
				    server->name,
				    keepalive_retries);
				deactivate_server(server);

			} else {
				log_printf(LOG_INFO,
				    "Server %s has timed out %d times, "
				    "retrying",
				    server->name,
				    server->retries);

				/* try for another keepalive */
				send_keepalive(server);

				/* create & schedule the timeout for this
				 * keepalive packet */
				if (!(new_task = malloc(sizeof(task_t)))) {
					perror("malloc");
					break;
				}

				new_task->time = time(NULL) + keepalive_timeout;
				new_task->operation = TASK_OP_KEEPALIVE_TIMEOUT;
				new_task->data1 = (int)server;

				log_printf(LOG_VDEBUG, "Adding a Keepalive "
				    "timeout for server %s",
				    ((struct server *)(task->data1))->name);
				list_add_ordered(&tasks, new_task, comp_tasks);
				log_printf(LOG_VDEBUG, "Task list now has %d "
				    "items", tasks.count);
			}

			break;

		case TASK_OP_REMOVAL: /* remove a service */

			log_printf(LOG_DEBUG, "Removal task raised");

			service = (struct service *) task->data1;
			server = service->server;

			/* single service removal */
			if (conditional_service_removal(service) > 0) {
				log_printf(LOG_DEBUG,
				    "Service (server %s, protocol %d, port "
				    "%d) has active connections, "
				    "scheduling another removal task",
				    service->server->name,
				    service->virtual_service->
					ipvs_service.protocol,
				    ntohs(service->port));

				/* schedule another removal event for later */
				if (!(new_task = malloc(sizeof(task_t)))) {
					perror("malloc");
					break;
				}

				new_task->time = time(NULL) + removal_interval;
				new_task->operation = TASK_OP_REMOVAL;
				new_task->data1 = task->data1;

				list_add_ordered(&tasks, new_task, comp_tasks);
				log_printf(LOG_VDEBUG, "Task list now has %d "
						"items", tasks.count);
			} else {
				delete_service_entry(service);
				if (!server->sd) {
					conditional_server_removal(server);
				}
			}
			break;
		}

		/* we're done with this task now */
		free(task);
	}
}

int get_timeout()
{
	if (tasks.head) {
		return (((task_t *)tasks.head->item)->time - time(NULL));
	} else {
		return 0;
	}
}

int unschedule_server(struct server *server)
{
	task_t *task;
	listitem **i;
	int ret = -1;

	assert(server);

	log_printf(LOG_VDEBUG, "Unscheduling events for server %s",
	                server->name);

	i = &(tasks.head);

	while (*i) {
		task = (task_t *)(*i)->item;
		/* find matching tasks for this server and remove */
		if (task->data1 == (int) server &&
		    (task->operation & 0x03) == task->operation) {

			listitem *tmp = *i;
			*i = (*i)->next;
			free(tmp);
			free(task);
			tasks.count--;
			ret = 0;
			continue;
		}
		i = &((*i)->next);
	}

	if (!ret)
		log_printf(LOG_VDEBUG, "Task list now has %d items",
				tasks.count);

	return ret;
}

void print_tasklist()
{
	listitem *i;
	task_t *task;

	i = tasks.head;

	log_printf(LOG_VDEBUG, "-- Tasklist dump at %d", time(NULL));

	while (i) {
		task = (task_t *)i->item;
		log_printf(LOG_VDEBUG,
		    "\tTask %p:"
		    "\top: %d"
		    "\ttime: %d"
		    "\tdata: %d",
		    task,
		    task->operation,
		    task->time,
		    task->data1);
		i = i->next;
	}

	log_printf(LOG_VDEBUG, "-- End Tasklist dump");
}

int comp_tasks(void *x, void *y)
{
	return ((task_t *)x)->time > ((task_t *)y)->time ? 1 : 0;

}

int server_inuse(struct server *server)
{
	listitem *i = services.head;
	int nservices = 0;

	while (i) {
		struct service *service = (struct service *)(i->item);
		if (service->server == server)
			nservices++;
		i = i->next;
	}
	return nservices;
}

