/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file perl.c
 *
 * Monitor plugin to interface to a perl script.
 * This plugin requires a single configuration tag:
 *  <file>/path/to/perlscript.pl</file>
 * Which specifies a per script to parse when then plugin is initialised.
 * When the load value is required, a get_load() sub is called in the
 * perl file, which should return a SCALAR, between 0 and 100, representing
 * the system health (100 is perfect health, 0 is unable to serve requests).
 *
 * The perl script is only parsed once, but the get_health sub is called
 * whenever necessary.
 */

#include <errno.h>

#include <EXTERN.h>
#include <perl.h>
#include <XSUB.h>

#include "plugin.h"
#include "utils.h"
#include "log.h"
#include "feedbackd-agent.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef call_pv
#define call_pv perl_call_pv
#endif

struct perl_monitor {
	struct monitor	base;
	char		*file;
	char		*args[3];
	PerlInterpreter	*perl;
};

/* if we have multiple plugins using a non-multiple perl library, we
 * need to free the interpreter after each use, so we need to consider
 * the perl_monitor->perl pointer invalidated between calls */
static int n_interpreters = 0;

static int config(struct monitor *monitor, char *name, char *value)
{
	struct perl_monitor *pm = (struct perl_monitor *)monitor;

	if (!streq(name, "file")) {
		pm->file = strdup(value);

	} else {
		plugin_printf(LOG_ERR, perl, "Unknown configuration "
				"directive %s", name);
		return -1;
	}

	return 0;
}

static int init(struct monitor *monitor)
{
	struct perl_monitor *pm = (struct perl_monitor *)monitor;
	struct stat statbuf;


	if (!pm->file) {
		plugin_printf(LOG_ERR, perl, "No file specified");
		return -1;
	}

	if (stat(pm->file, &statbuf)) {
		plugin_printf(LOG_ERR, perl, "File %s doesn't exist");
		return -1;
	}

	pm->args[0] = "feedbackd-agent";
	pm->args[1] = pm->file;
	pm->args[2] = NULL;

	n_interpreters++;

#ifdef PERL_MULTIPLICITY
	pm->perl = perl_alloc();
	PERL_SET_CONTEXT(pm->perl);
	perl_construct(pm->perl);
	perl_parse(pm->perl, NULL, 2, pm->args, NULL);
#else

	if (n_interpreters == 1) {
		/* this is the first perl monitor.
		 * Establish the single interpreter */
		pm->perl = perl_alloc();
		perl_construct(pm->perl);
		perl_parse(pm->perl, NULL, 2, pm->args, NULL);

	/* only warn once */
	} else if (n_interpreters == 2) {
		plugin_printf(LOG_WARN, perl, "Multiple perl plugins "
				"defined, but the perl interpreter "
				"does not support multiplicity. "
				"Continuing with less efficient "
				"monitor loading");
	}
#endif
	return 0;
}

static int get_health(struct monitor *monitor)
{
	struct perl_monitor *pm = (struct perl_monitor *)monitor;
	int health = 0, count;
	dSP;
	
#ifdef PERL_MULTIPLICITY
	PERL_SET_CONTEXT(pm->perl);
#else
	if (n_interpreters > 1) {
		if (pm->perl) {
			perl_destruct(pm->perl);
			perl_free(pm->perl);
		}
		pm->perl = perl_alloc();
		perl_construct(pm->perl);
		perl_parse(pm->perl, NULL, 2, pm->args, NULL);
	}
#endif

	plugin_printf(LOG_DEBUG, perl, "measuring health with "
		                  "script %s", pm->file);
	
	ENTER;
	SAVETMPS;

	PUSHMARK(SP);
	PUTBACK;

	count = call_pv("get_health", G_NOARGS | G_SCALAR);
	SPAGAIN;
	
	if (count == 1)
		health = POPi;
	
	PUTBACK;
	FREETMPS;
	LEAVE;

#ifndef PERL_MULTIPLICITY
	if (n_interpreters > 1) {
		perl_destruct(pm->perl);
		perl_free(pm->perl);
		pm->perl = NULL;
	}
#endif
	
	return health;
}

static int destroy(struct monitor *monitor)
{
	struct perl_monitor *pm = (struct perl_monitor *)monitor;

	if (pm->perl) {
		perl_destruct(pm->perl);
		perl_free(pm->perl);
	}
	free(pm->file);
	free(pm);
	return 0;
}

struct monitor *create_monitor(struct service *service)
{
	struct perl_monitor *monitor;

	monitor = malloc(sizeof(*monitor));
	if (!monitor)
		return NULL;
	memset(monitor, 0, sizeof(monitor));

	monitor->base.config     = config;
	monitor->base.init       = init;
	monitor->base.get_health = get_health;
	monitor->base.destroy    = destroy;

	return (struct monitor *)monitor;
}
