/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2007 Jeremy Kerr
 *
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file cpuload.c
 *
 * Plugin to monitor the CPU load, and report it as system health. CPU load is
 * measured from /proc/stat when get_load is called, and calculated using
 * previously obtained values.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <poll.h>

/* default timeout value, in seconds */
#define DEFAULT_TIMEOUT 2

#include "plugin.h"
#include "utils.h"
#include "log.h"
#include "feedbackd-agent.h"

struct exec_monitor {
	struct monitor	base;
	int		timeout;
	char *		command;
};

static int null_fd = -1;

static int config(struct monitor *monitor, char *name, char *value)
{
	struct exec_monitor *em = (struct exec_monitor *)monitor;

	if (streq(name, "command")) {
		em->command = strdup(value);

	} else if (streq(name, "timeout")) {
		if (parse_number(&em->timeout, value, INT_MAX, "timeout"))
			return -1;

	} else {
		plugin_printf(LOG_ERR, exec, "Unknown configuration "
				"directive %s", name);
		return -1;
	}

	return 0;
}

static int init(struct monitor *monitor)
{
	struct exec_monitor *em = (struct exec_monitor *)monitor;
	if (!em->command) {
		plugin_printf(LOG_ERR, constant, "No command specified");
		return -1;
	}

	if (null_fd == -1) {
		null_fd = open("/dev/null", O_WRONLY);
		if (null_fd < 0) {
			plugin_printf(LOG_ERR, exec, "Can't open /dev/null?");
			return -1;
		}
	}


	return 0;
}

static int get_health(struct monitor *monitor)
{
	struct exec_monitor *em = (struct exec_monitor *)monitor;
	int health, pid;
	int pipefds[2];
	int status;

	if (pipe(pipefds)) {
		plugin_printf(LOG_ERR, exec, "Error creating pipe: %s",
				strerror(errno));
		return 0;
	}

	health = 0;

	pid = fork();
	if (pid < 0) {
		plugin_printf(LOG_ERR, exec, "Error creating new process: %s",
				strerror(errno));
		goto out_close;
	}

	/* child: execute the command, sending output through the pipe */
	if (!pid) {
		/* stderr to /dev/null */
		if (dup2(null_fd, STDERR_FILENO) < 0)
			exit(EXIT_FAILURE);

		/* stdout to the pipe */
		if (dup2(pipefds[1], STDOUT_FILENO) < 0)
			exit(EXIT_FAILURE);

		execlp("/bin/sh", "/bin/sh", "-c", em->command, NULL);
		exit(EXIT_FAILURE);

	/* parent: read from child's stdout, but don't block waiting for it
	 * for longer than the timeout */
	} else {
		int rc;
		char buf[1024], *end;
		struct pollfd pollfds[1];

		pollfds[0].fd = pipefds[0];
		pollfds[0].events = POLLIN;
		pollfds[0].revents = 0;

		rc = poll(pollfds, 1, em->timeout * 1000);

		if (rc != 1 || pollfds[0].revents & POLLERR) {
			plugin_printf(LOG_WARN, exec, "Error waiting for "
					"output of command %s",
					em->command);
			goto out_wait;
		}

		memset(buf, 0, sizeof(buf));

		if (read(pipefds[0], buf, sizeof(buf) - 1) <= 0) {
			plugin_printf(LOG_WARN, exec, "Error reading output "
					"of command %s", em->command);
			goto out_wait;
		}
		close(pipefds[0]);

		errno = 0;
		health = strtoul(buf, &end, 10);
		if (end == buf || errno == ERANGE) {
			plugin_printf(LOG_ERR, exec, "program %s returned "
					"invalid output (begins with: '%s')",
					em->command, buf);
			health = 0;
		}

		/* clear the output buffer, we don't want the child blocked
		 * on a write */
		while (read(pipefds[0], buf, sizeof(buf)) > 0)
			;
	}


out_wait:
	/* we may not catch the process here, but we don't want to get stuck
	 * waiting. We'll catch any missed children next time around */
	waitpid(-1, &status, WNOHANG);

out_close:
	close(pipefds[0]);
	close(pipefds[1]);

	return health;
}

static int destroy(struct monitor *monitor)
{
	struct exec_monitor *cm = (struct exec_monitor *)monitor;
	free(cm->command);
	free(cm);
	return 0;
}

struct monitor *create_monitor(struct service *service)
{
	struct exec_monitor *monitor;

	monitor = malloc(sizeof(*monitor));
	if (!monitor)
		return NULL;
	memset(monitor, 0, sizeof(monitor));

	monitor->base.config = config;
	monitor->base.init = init;
	monitor->base.get_health = get_health;
	monitor->base.destroy = destroy;
	monitor->timeout = DEFAULT_TIMEOUT;

	return (struct monitor *)monitor;
}

