/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file cpuload.c
 *
 * Plugin to monitor the CPU load, and report it as system health. CPU load is
 * measured from /proc/stat when get_load is called, and calculated using
 * previously obtained values.
 */

#include "plugin.h"

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

struct cpuload_monitor {
	struct monitor	base;
};

#ifdef __linux__

#elif defined(__FreeBSD__)

#include <sys/param.h>
#include <sys/linker.h>
#include <sys/dkstat.h>

#include <fcntl.h>
#include <kvm.h>

static kvm_t *kd;
static struct nlist nlst[] = {
	{ "_cp_time" },
	{ NULL }
};

#else
#error No OS definition found for cpu load monitoring. \
       Add one of -D__linux__ or -D__FreeBSD__ to CFLAGS.
#endif

static uint64_t lastidlesample;
static uint64_t lastusagesample;
static uint64_t lasthealthvalue;


#ifdef __linux__
static void getusage(uint64_t *usage, uint64_t *idle)
{
	/* cpu times read:
	 *  0: user
	 *  1: nice
	 *  2: system
	 *  3: idle
	 */
	uint64_t value[4];
	FILE *procfile;

	/* /proc/stat needs to be reopened for each measurement, otherwuse
	 * the values are not updated */
	procfile = fopen("/proc/stat", "r");
	if (!procfile) {
		plugin_printf(LOG_ERR, cpuload, "Could not open /proc/stat");
		return;
	}
	
	if (fscanf(procfile, "cpu  %llu %llu %llu %llu",
	       &(value[0]), &(value[1]), &(value[2]), &(value[3])) == 4) {

		*usage = value[0] + value[1] + value[2];
		*idle  = value[3];
	} else {
		plugin_printf(LOG_ERR, cpuload, "Could not read CPU load");
	}

	fclose(procfile);
}

#elif defined(__FreeBSD__)
static void getusage(uint64_t *usage, uint64_t *idle)
{
	long value[CPUSTATES];
	unsigned int bytes;
	
	bytes = sizeof(value);
	if (bytes != kvm_read(kd, nlst[0].n_value, value, bytes)) {
		perror("kvm_read");
		exit(1);
	}

	*usage = value[CP_USER] + value[CP_NICE] +
		 value[CP_SYS] + value[CP_INTR];
	*idle =  value[CP_IDLE];
}
#endif

/* since cpu load is global, we have one measuring monitor (which does the
 * load calculation) only. Any other montiors just use the last values.
 *
 * If this becomes null (the measuring monitor went away), the next monitor
 * to call get_health() becomes the measuring monitor
 */
static struct cpuload_monitor *measuring_monitor;

static int get_health(struct monitor *monitor)
{
	struct cpuload_monitor *cm = (struct cpuload_monitor *)monitor;

	if (measuring_monitor == NULL)
		measuring_monitor = cm;

	if (measuring_monitor == cm) {
		uint64_t idlesample, usagesample, total;
		int health;

		getusage(&usagesample, &idlesample);

		total = usagesample + idlesample;

		/* sanity check if no time has elapsed. */
		if (total - (lastusagesample + lastidlesample) == 0) {
			health = lasthealthvalue;
		} else {
			health = 100 * (idlesample - lastidlesample) /
			     (total - (lastusagesample + lastidlesample));
		}

		/* prevent 100% cpu usage from removing the machine from the
		 * cluster */
		if (!health)
			health = 1;

		lastusagesample = usagesample;
		lastidlesample = idlesample;
		lasthealthvalue = health;
	}
	
	return lasthealthvalue;
}

static void init_measurements(struct monitor *monitor)
{
	static int inited;

	if (inited++)
		return;

#ifdef __FreeBSD__

	if (!(kd = kvm_open(NULL, NULL, NULL, O_RDONLY, "kvm_open"))) {
		perror("kvm_open");
	}
	
	if (kvm_nlist(kd, nlst)) {
		perror("kvm_nlist");
	}

#endif
	/* we need to read the load values on initialisation, so that
	 * there's a sensible value for the first invocation of get_health */
	getusage(&lastusagesample, &lastidlesample);
	lasthealthvalue = get_health(monitor);
}

static int init(struct monitor *monitor)
{
	init_measurements(monitor);
	return 0;
}

static int destroy(struct monitor *monitor)
{
	struct cpuload_monitor *cm = (struct cpuload_monitor *)monitor;

	if (measuring_monitor == cm)
		measuring_monitor = NULL;

	return 0;
}

struct monitor *create_monitor(struct service *service)
{
	struct cpuload_monitor *monitor;

	monitor = malloc(sizeof(*monitor));
	if (!monitor)
		return NULL;
	memset(monitor, 0, sizeof(monitor));

	monitor->base.init       = init;
	monitor->base.get_health = get_health;
	monitor->base.destroy    = destroy;

	return (struct monitor *)monitor;
}
