/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 *
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file necp_handler.h
 *
 * Functions to handle necp data from a server.
 */

#ifndef __HAVE_NECP_HANDLER_H
#define __HAVE_NECP_HANDLER_H 1

#include "feedbackd-master.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

/**
 * Handle an incoming NECP packet. The packet should already have had its
 * protocol identifier and version checked through necp_listener
 *
 * @param serverp A pointer to the server object that the packet is from
 * @param packet A pointer to the packet data
 * @param size The size of the packet
 */
void handle_necp_data(struct server *serverp, char *packet, int size);

/**
 * Send a NECP_KEEPALIVE request to a server. This will be called by the
 * scheduler when it is time for a load update. All services provided by that
 * server will be included in the request.
 *
 * @param serverp The server to query.
 */
void send_keepalive(struct server *serverp);

/**
 *  Close the connection to a server. This should be called when a server has
 *  been disconnected. This function does the following:
 *  * Unschedules any keepalive / keepalive timeout events for active services
 *  * Quiesces any services that are open for this server
 *  * Tells the necp_listener to close the socket descriptor
 *  * If the server has no services associated, the server is removed from the
 *    the server list (otherwise this will be done when the last service is
 *    removed)
 *
 *
 *  @param serverp The server to deactivate
 */
int deactivate_server(struct server *serverp);

/**
 * Delete the entry for a service. Only do this if this service is not
 * required by any events.
 *
 * @param service the entry to delete
 */
void delete_service_entry(struct service *service);

/**
 * Handle an incoming NECP INIT packet
 *
 * @param server the server the packet is from
 * @param packet the packet that has been received
 * @return Non-zero if there were errors processing the packet, zero otherwise
 */
int handle_necp_init(struct server *server, char *packet);

/**
 * Handle an incoming NECP START packet
 *
 * @param server the server the packet is from
 * @param packet the packet that has been received
 * @return Non-zero if there were errors processing the packet, zero otherwise
 */
int handle_necp_start(struct server *server, char *packet);

/**
 * Handle an incoming NECP STOP packet
 *
 * @param server the server the packet is from
 * @param packet the packet that has been received
 * @return Non-zero if there were errors processing the packet, zero otherwise
 */
int handle_necp_stop(struct server *server, char *packet);

/**
 * Handle an incoming NECP KEEPALIVE ACK packet
 *
 * @param server the server the packet is from
 * @param packet the packet that has been received
 * @return Non-zero if there were errors processing the packet, zero otherwise
 */
int handle_necp_keepalive_ack(struct server *server, char *packet);

/**
 * Find a service with that matches given parameters.
 *
 * @param server The server that provices the service
 * @param protocol the protocol ID (TCP or UDP) that the service uses
 * @param port the port that the service uses
 * @return a pointer to the service if it exists, NULL otherwise
 */
struct service *find_service(struct server *server, uint8_t protocol,
		uint16_t port);

int conditional_service_removal(struct service *service);
int conditional_server_removal(struct server *server);
#endif /* HAVE_NECP_HANDLER */

