/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 *
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file necp_client.c
 */
#include "necp_client.h"
#include "necp.h"

#include <assert.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <netinet/in.h>

#include "log.h"
#include "plugins/plugin.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/**
 * The biggest packet we're willing to malloc for. (In payload units)
 */
#ifndef MAX_PAYLOAD_N
#define MAX_PAYLOAD_N 10
#endif


static struct service *find_service(uint8_t protocol, uint16_t port)
{
	listitem *i;

	list_for_each(services, i) {
		struct service *service = (struct service *)i->item;
		if (service->protocol == protocol &&
				service->port == port)
			return service;
	}

	return NULL;
}


int necp_init(int sd)
{
	struct necp_header *header;
	struct necp_payload_unit *payload;
	char *packet;
	int x;

	/* create a necp packet */
	if (!(packet = malloc(NECP_PACKET_SIZE(1)))) {
		perror("malloc");
		return -1;
	}

	memset(packet, 0, NECP_PACKET_SIZE(1));

	header = (struct necp_header *)packet;
	payload = (struct necp_payload_unit *)(packet + sizeof(*header));

	/* set fields for a NECP INIT */
	header->payload_len = 32;
	header->flags = NECP_F_Basic_Payload;
	header->opcode = NECP_INIT;
	header->request_id = request_id = rand();

	/* no auth at the moment, set data{0,1,2} fields to 0 */
	payload->data0 = payload->data1 = payload->data2 = 0;

	log_printf(LOG_DEBUG, "Sending NECP_INIT");

	x = send_packet(sd, packet);

	free(packet);

	return x;
}


int necp_start(int sd)
{
	struct necp_header *header;
	struct necp_payload_unit *payload;
	char *packet;
	listitem *i;
	struct service *service;
	int x;

	/* cerate a necp packet */
	if (!(packet = malloc(NECP_PACKET_SIZE(services.count)))) {
		perror("malloc");
		return -1;
	}

	memset(packet, 0, NECP_PACKET_SIZE(services.count));

	header = (struct necp_header *)packet;
	payload = (struct necp_payload_unit *)(packet + sizeof(*header));

	/* standard header settings */
	header->payload_len = services.count * sizeof(*payload);
	header->flags = NECP_F_Basic_Payload;
	header->opcode = NECP_START;
	header->request_id = request_id = rand();

	/* set the payload for all services configured */
	x = 0;
	for (i = services.head; i; i = i->next) {
		service = (struct service *)i->item;
		payload = (struct necp_payload_unit *)
			(packet + sizeof(*header) +
			 x++ * sizeof(*payload));

		payload->data0 = htonl(service->forwarding);
		payload->data1 = htonl(service->protocol);
		payload->data2 = htons(service->port);
		payload->data2 = htonl(payload->data2);
		service->monitor->get_health(service->monitor);
	}

	log_printf(LOG_DEBUG, "Sending NECP_START");

	x = send_packet(sd, packet);

	free(packet);

	return x;
}

int necp_handle_start_ack(char *packet)
{
	struct necp_payload_unit *payload;
	struct necp_header *header;
	unsigned int packetsize, i;

	assert(packet);

	header = (struct necp_header *)packet;

	if (!(header->flags & NECP_F_Error))
		return 0;

	packetsize = sizeof(*header) + header->payload_len;

	/* check for rejected services, and warn */
	for (i = sizeof(*header); i < packetsize; i += sizeof(*payload)) {
		struct service *service;
		uint16_t port;

		payload = (struct necp_payload_unit *)(packet + i);
		port = htonl(payload->data2);

		service = find_service(htonl(payload->data1), htons(port));
		if (!service) {
			log_printf(LOG_WARN, "start ack for unknown service?");
			continue;
		}

		log_printf(LOG_WARN, "Service %s rejected by master, "
					"disabling", service->name);

		list_remove(&services, service);
	}

	if (list_empty(services)) {
		log_printf(LOG_ERR, "No services accepted by the master");
		return -1;
	}

	return 0;
}


int necp_stop(int sd)
{
	struct necp_header *header;
	struct necp_payload_unit *payload;
	char *packet;
	listitem *i;
	struct service *service;
	int x;

	/* create a necp packet */
	if (!(packet = malloc(NECP_PACKET_SIZE(services.count)))) {
		perror("malloc");
		return -1;
	}

	memset(packet, 0, NECP_PACKET_SIZE(services.count));

	header = (struct necp_header *)packet;
	payload = (struct necp_payload_unit *)(packet + sizeof(*header));

	/* set fields for a NECP STOP */
	header->payload_len = services.count * sizeof(*payload);
	header->flags = NECP_F_Basic_Payload;
	header->opcode = NECP_STOP;
	header->request_id = request_id = rand();

	/* stop all configured services */
	x = 0;
	for (i = services.head; i; i = i->next) {
		service = (struct service *)i->item;
		payload = (struct necp_payload_unit *)
			  (packet + sizeof(*header) +
			   x++ * sizeof(*payload));
		payload->data0 = htonl(service->forwarding);
		payload->data1 = htonl(service->protocol);
		payload->data2 = ntohs(service->port);
		payload->data2 = htonl(payload->data2);
	}

	log_printf(LOG_DEBUG, "Sending NECP_STOP");

	x = send_packet(sd, packet);

	free(packet);

	return x;
}

int necp_keepalive(int sd, char *packet)
{
	struct necp_payload_unit *payload;
	struct necp_header *header;
	unsigned int packetsize, i;
	struct service *service;

	assert(sd > 0);
	assert(packet);

	header = (struct necp_header *)packet;

	packetsize = sizeof(*header) + header->payload_len;

	/* set the data3 field to the load of each configured service */
	for (i = sizeof(*header); i < packetsize; i += sizeof(*payload)) {
		payload = (struct necp_payload_unit *)(packet + i);
		uint16_t port = htonl(payload->data2);

		service = find_service(htonl(payload->data1), htons(port));
		if (!service) {
			log_printf(LOG_WARN, "Requested health for "
					"unknown service?");
			continue;
		}

		payload->data3 = htonl(service->monitor->get_health(
					service->monitor));
	}

	/* change the header to an ACK */
	header->opcode = NECP_KEEPALIVE_ACK;
	header->flags = NECP_F_Basic_Payload;

	log_printf(LOG_DEBUG, "Sending NECP_KEEPALIVE_ACK");

	return send_packet(sd, packet);
}

int send_packet(int sd, char *packet)
{
	struct necp_header *header;
	unsigned int packetsize;
	int bw;

	assert(sd > 0);
	assert(packet);

	header = (struct necp_header *)packet;

	/* set global header fields for all packets */
	header->protocol_identifier = NECP_PID;
	header->version = NECP_VERSION;

	/* will require changing when authentication is added */
	header->seq_num = 0;

	/* grab the packet size before we mangle the header */
	packetsize = header->payload_len + sizeof(*header);

	/* prepare the header */
	prepare_header_out(header);

	/* send... */
	if ((bw = write(sd, (void *)packet, packetsize) != packetsize)) {
		log_printf(LOG_ERR,
		   "Error sending packet: write() returned %d "
		   "(expected %d)", bw, packetsize);
		if (errno == EPIPE)
			return -2;
		else
			return -1;
	}

	return 0;
}

int data_received(int sd)
{
	struct necp_header *header;
	char *buf;
	unsigned int packetsize, ret;
	ssize_t rc;

	assert(sd > 0);

	if (!(buf = malloc(sizeof(struct necp_header)))) {
		perror("malloc");
		return -1;
	}

	rc = read(sd, buf, sizeof(*header));

	/* read data from socket into buf */
	if (rc != sizeof(*header)) {
		perror("read");
		log_printf(LOG_WARN, "Incorrect number of bytes read: %d "
		              "closing connection", rc);
		return -2;
	}

	/* set up the necp header */
	header = (struct necp_header *) buf;
	prepare_header_in(header);

	/* check for length sanity */
	if (header->payload_len > MAX_PAYLOAD_N *
			sizeof(struct necp_payload_unit)) {
		log_printf(LOG_WARN, "Packet was too large "
		              "(%d payload units, max is %d)",
			      header->payload_len /
			      sizeof(struct necp_payload_unit),
			      MAX_PAYLOAD_N);
		while (read(sd, NULL, 1024) > 0);
		return -1;
	}


	/* determine the size of the whole packet from the header */
	packetsize = sizeof(*header) + header->payload_len;

	/* allocate memory for the packet */
	if (!(buf = realloc(buf, packetsize))) {
		perror("realloc");
		return -1;
	}

	/* need to reassign header as buf may have moved */
	header = (struct necp_header *) buf;

	/* read the rest of the packet */
	rc = read(sd, buf + sizeof(*header), header->payload_len);

	if (rc != header->payload_len) {
		perror("read");
		log_printf(LOG_WARN, "Incorrect number of bytes read: %d "
		              "closing connection", rc);
		free(buf);
		return -2;
	}

	ret = 0;

	/* action depends on the packet's opcode */
	switch (header->opcode) {
		case NECP_INIT_ACK:
			log_printf(LOG_DEBUG, "Got NECP_INIT_ACK");
			ret = necp_start(sd);
			break;

		case NECP_START_ACK:
			log_printf(LOG_DEBUG, "Got NECP_START_ACK");
			ret = necp_handle_start_ack(buf);
			break;

		case NECP_KEEPALIVE:
			log_printf(LOG_DEBUG, "Got NECP_KEEPALIVE");
			ret = necp_keepalive(sd, buf);
			break;

		case NECP_STOP_ACK:
			log_printf(LOG_DEBUG, "Got NECP_STOP_ACK");
			/* only stop if we're shutting down */
			if (stop_service) {
				running = 0;
			}
			break;
	}

	free(buf);

	return ret;
}
