/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
 * @file necp.h
 * Standard NECP definions for both client and server
 * use.
 *
 */

#ifndef __HAVE_NECP_H
#define __HAVE_NECP_H 1

#include <stdint.h>

#define NECP_PORT	3262	/**< The port to listen for NECP data on */
#define NECP_VERSION	0x1	/**< Version of NECP implemented */
#define NECP_PID	0x414a	/**< NECP protocol identifier */

/**
 * Macro to give the size (in bytes, of header & payload) of a necp packet
 * 
 * @param x the number of payload units in the packet
 */
#define NECP_PACKET_SIZE(x) (sizeof(struct necp_header) + \
		x * sizeof(struct necp_payload_unit))

/**
 * Number of keepalive retries before a server is considered dead
 */
#define NECP_RETRIES 3

/**
 * Options for the opcode field in the necp necp_header
 */
enum necp_opcode {
	NECP_NOOP		= 0x00, /**< No operation */
	NECP_INIT		= 0x01, /**< Initialise the NECP session */
	NECP_INIT_ACK		= 0x02, /**< ACK for initialisation */
	NECP_KEEPALIVE		= 0x03, /**< Status request */
	NECP_KEEPALIVE_ACK	= 0x04, /**< Status reply */
	NECP_START		= 0x05, /**< Begin forwarding requests */
	NECP_START_ACK		= 0x06, /**< Begin ACK */
	NECP_STOP		= 0x07, /**< Stop forwarding requests */
	NECP_STOP_ACK		= 0x08  /**< Stop ACK */
};

/**
 * Forwarding type options
 */
typedef enum {
	NECP_FT_DROUTE		= 0x01, /**< Direct routing */
	NECP_FT_TUNNEL		= 0x02, /**< GRE/Tunnelling */
	NECP_FT_MASQ		= 0x03	/**< NAT/Masquerading */
} necp_ftype;

/**
 * NECP header unit.
 */
struct necp_header {
	uint16_t protocol_identifier;	/**< Protocol identifier
					     (always "AJ") */
	uint16_t flags;			/**< Flags field */
	uint8_t  version;		/**< NECP Version */
	uint8_t  opcode;		/**< Operation */
	uint16_t request_id;		/**< The (sender-specific) request
					     identifier */
	uint64_t seq_num;		/**< The sequence number of this
					     packet */
	uint32_t payload_len;		/**< Length of the payload (not inluding
					     the header) */
} __attribute__((packed));

/**
 * NECP payload unit
 */
struct necp_payload_unit {
	uint32_t data0;	/**< Data element 0 */
	uint32_t data1;	/**< Data element 1 */
	uint32_t data2;	/**< Data element 2 */
	uint32_t data3;	/**< Data element 3 */
	uint32_t data4;	/**< Data element 4 */
	uint32_t data5;	/**< Data element 5 */
	uint32_t data6;	/**< Data element 6 */
	uint32_t data7;	/**< Data element 7 */
} __attribute__((packed));

/* flags */
#define NECP_F_Basic_Payload			0x0001 /**< NECP Flag: Basic (fixed length)
							               payloads */
#define NECP_F_Auth_Credential_Provided		0x0002 /**< NECP Flag: This packet contains
							               a credential */
#define NECP_F_Error				0x0004 /**< NECP Flag: Error */
#define NECP_F_Protocol_Version_Mismatch	0x0008 /**< NECP Flag: Wrong protocol
							               version */
#define NECP_F_Auth_Required			0x0010 /**< NECP Flag: Auth required for
							               this session */
#define NECP_F_Bad_Sequence_Number		0x0020 /**< NECP Flag: Bad sequence number
							               sent */

/**
 * Read an incoming header and perform endian conversion
 * 
 * @param header a pointer to a NECP header
 */
void prepare_header_in(struct necp_header *header);

/**
 * Read an outgoing header and perform endian conversion
 * 
 * @param header a pointer to a NECP header
 */
void prepare_header_out(struct necp_header *header);

#endif /* end __HAVE_NECP_H */
