/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file log.c
 */

#include "log.h"

#include <time.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

/* default values - used before and during configuration, after which point the
 * user-defined values will be used
 */
unsigned int loglevel_ = LOG_WARN;
static FILE *logstream;

extern struct tm *gmtime_r(const time_t *timep, struct tm *result);
extern struct tm *localtime_r(const time_t *timep, struct tm *result);

/* if fileno is a macro (BSD), don't define */
#ifndef fileno
extern int fileno(FILE *__stream);
#endif

void log_printf(loglevel level, const char *format, ...)
{
	va_list ap;
	FILE *stream;

	stream = logstream ? logstream : stderr;

	if (level <= loglevel_) {
		char s[30];
		time_t t1;
		struct tm t2;

		time(&t1);
		localtime_r(&t1, &t2);
		memset(s, 0, sizeof(s));
		strftime(s, 30, TIMEFMT, &t2);

		va_start(ap, format);
		fprintf(stream, s);
		vfprintf(stream, format, ap);
		fprintf(stream, "\n");
		fflush(stream);
		va_end(ap);
	}
}

void log_stop(void)
{
	if (logstream && (logstream != stdout || logstream != stderr)) {
		fclose(logstream);
	}
}

static void setdup(void)
{
	if (logstream != stderr)
		dup2(fileno(logstream), STDERR_FILENO);
	if (logstream != stdout)
		dup2(fileno(logstream), STDOUT_FILENO);
}


int log_setlevel(const char *level)
{
	unsigned int l1, i;
	const char* levels[] =
		{"CRIT", "ERR", "WARN", "INFO", "DEBUG", "VDEBUG"};

	l1 = strlen(level);

	for (i = LOG_CRIT; i <= LOG_VDEBUG; i++) {
		unsigned int l2 = strlen(levels[i-1]);
		if (!strncmp(level, levels[i-1], l1 < l2 ? l1 : l2)) {
			loglevel_ = i;
			return 0;
		}
	}

	return -1;
}

int log_setfile(const char *file)
{

	if (!strcmp(file, "-"))  {
		logstream = stdout;

	} else {
		logstream = fopen(file, "a");
		if (!logstream) {
			log_printf(LOG_ERR, "Could not open log file "
					"%s: %s", file, strerror(errno));
			return -1;
		}
	}
	setdup();
	return 0;
}
