/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file linkedlist.h
 * A simple implementation of a singly-linked list of pointers to any object.
 */

#ifndef __HAVE_LINKEDLIST_H
#define __HAVE_LINKEDLIST_H 1

#define list_for_each(list,i) for (i = list.head; i; i = i->next)

#define list_empty(list) (list.count == 0)

/**
 * A list item
 */
struct listitem_ {
	void *item;		/**< The item in the list */
	struct listitem_ *next;	/**< A pointer to the next item */
};

/**
 * A list item
 */
typedef struct listitem_ listitem;

/**
 * A list - a pointer to the first list item
 */
/* typedef listitem *list; */
typedef struct {
	listitem *head;	/**< The head of the list */
	int count;	/**< Number of items in the list */
} list;

/**
 * Initialise a list
 * @param list A list to initialise
 */
void list_init(list *list);

/**
 * Add an item onto the list. New items are added to the head of the list
 * @param list The list to add to
 * @param item A pointer to the item to add
 * @return zero if the item was added sucessfully, false otherwise
 */
int list_add(list *list, void *item);

/**
 * Add an item onto the list, in a specific order. For this function to work
 * as expected, the list must already be ordered. The comp function provides
 * the comparison code, and is invoked with two list items to compare.
 * comp should return non-zero if its first argument is greater than its
 * second, zero otherwise.
 *
 * For example, this function will give an ordered list of characters:
 * 
 * <pre>return *(char *)x > *(char *)y ? 1 : 0;</pre>
 * 
 * @param list A pointer to the list to add to
 * @param item A pointer to the item to add
 * @param comp Function to provide comparison code
 * @return zero if the item was added sucessfully, false otherwise
 */
int list_add_ordered(list *list, void *item, int comp(void *, void *));

/**
 * Remove an item from the list. You'd probably want to free() the item
 * after removal. If item does not exist in the list, the list remains
 * unchanged.
 * 
 * @param list A pointer to the list to remove from
 * @param item A pointer to the item to remove
 * @return zero if the item was removed, non-zero otherwise (the list has not
 *         been altered)
 */
int list_remove(list *list, void *item);

#endif /* __HAVE_LINKEDLIST_H */
