/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
 * @file ipvs_interface.h
 *
 * Provides the interfave to the ipvs system, through libipvs
 * When setting the parameters for a service, not all of the fields
 * of the user rule are set. This may cause problems with future
 * versions of ipvs?
 */
#ifndef __HAVE_IPVS_SERVICE_H
#define __HAVE_IPVS_SERVICE_H 1


#include "feedbackd-master.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

/* if we don't have the ipvs headers around, fake up out own structure */
#ifdef WITH_IPVS
#include "libipvs/libipvs.h"
#else

#define IP_VS_SCHEDNAME_MAXLEN	16

typedef struct {
	/* virtual service addresses */
	uint16_t		protocol;
	uint32_t		addr;		/* virtual ip address */
	uint16_t		port;
	uint32_t		fwmark;		/* firwall mark of service */

	/* virtual service options */
	char			sched_name[IP_VS_SCHEDNAME_MAXLEN];
} ipvs_service_t;
#endif

struct service;

/**
 * Set the ipvs interface to dry-run mode - no changes
 * will be made to ipvs tables, although the tables will still be
 * read to get the number of connections for a service.
 * @param dryrun_flag Non-zero to put the interface in dry-run mode
 */
void ipvsif_set_dryrun(int dryrun_flag);

/**
 * Set the weight of the service.
 * 
 * @param service The service
 */
int ipvsif_set_weight(struct service *service);

/**
 * Add a service to the ipvs table
 * 
 * @param service The service to add
 */
int ipvsif_add_service(struct service *service);

/**
 * Remove a service from the table
 * 
 * @param service The service to remove
 */
int ipvsif_remove_service(struct service *service);

/**
 * Get the number of active connections for a service. services should only be
 * rmoved from the ipvs tables (with remove_service() ) is the number of
 * connections is zero.
 * 
 * @param service the service to query
 */
int ipvsif_get_service_conns(struct service *service);

/**
 * Initialise the ipvs interface
 *
 * @return 0 on success
 */
int ipvsif_init(void);

/**
 * Add virtual services to ipvs tables
 *
 * @return 0 on success
 */
int ipvsif_init_virtual_services(void);

/**
 * Close the ipvs interface
 */
void ipvsif_close(void);

#endif
