/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file ipvs_interface.c
 */

#include "feedbackd-master.h"
#include "log.h"
#include "ipvs_interface.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <errno.h>

#ifdef WITH_IPVS
#define PROCFILE "/proc/net/ip_vs_conn"
#include "libipvs/libipvs.h"
#endif

static int dryrun = 0;

int ipvsif_init(void)
{
#ifdef WITH_IPVS
	int rc;
	
	rc = ipvs_init();

	if (rc) 
		log_printf(LOG_ERR, "ipvs init error: %s",
				ipvs_strerror(errno));

	return rc;
#else
	return 0;
#endif
}

void ipvsif_set_dryrun(int dryrun_flag)
{
	dryrun = dryrun_flag ? -1 : 0;
}

#ifdef WITH_IPVS
static inline void ipvsif_init_dest(ipvs_dest_t *dest, struct service *service)
{
	dest->addr = service->server->address.s_addr;
	dest->port = service->port;

	switch (service->forwarding) {
	case NECP_FT_DROUTE:
		dest->conn_flags = IP_VS_CONN_F_DROUTE;
		break;
	case NECP_FT_TUNNEL:
		dest->conn_flags = IP_VS_CONN_F_TUNNEL;
		break;
	case NECP_FT_MASQ:
		dest->conn_flags = IP_VS_CONN_F_MASQ;
		break;
	}
}
#endif

int ipvsif_init_virtual_services(void)
{
#ifdef WITH_IPVS
	listitem *i;

	for (i = virtual_services.head; i; i = i->next) {
		struct virtual_service *service =
			(struct virtual_service *)i->item;
		if (ipvs_add_service(&service->ipvs_service)) {
			if (errno == EEXIST) {
				log_printf(LOG_INFO, "Virtual service %s "
						"already in ipvs tables.",
						service->name);
			} else {
				log_printf(LOG_INFO, "Virtual Service addition "
						"failed for service %s: %s",
						service->name,
						ipvs_strerror(errno));
				return -1;
			}
		}
	}
#endif
	return 0;
}

int ipvsif_set_weight(struct service *service)
{
#ifdef WITH_IPVS
	ipvs_dest_t dest;
	
	if (dryrun) return 0;

	memset (&dest, 0, sizeof(dest));

	ipvsif_init_dest(&dest, service);

	dest.weight = service->current_weight;

	if (ipvs_update_dest(&service->virtual_service->ipvs_service, &dest)) {
		log_printf(LOG_ERR, "IPVS error during weight update: %s",
		              ipvs_strerror(errno));
		return -1;
	}

#endif
	return 0;
}

int ipvsif_add_service(struct service *service)
{
#ifdef WITH_IPVS
	ipvs_dest_t dest;
	int rc;
	
	if (dryrun) return 0;

	memset (&dest, 0, sizeof(dest));

	ipvsif_init_dest(&dest, service);

	dest.weight = 0;

	rc = ipvs_add_dest(&service->virtual_service->ipvs_service, &dest);

	if (rc != 0 && errno != EEXIST) {
		log_printf(LOG_ERR, "IPVS error during RS addition: %s",
		              ipvs_strerror(errno));
		return -1;
	}

#endif
	return 0;
}

int ipvsif_remove_service(struct service *service)
{
#ifdef WITH_IPVS
	ipvs_dest_t dest;
	int rc;

	if (dryrun)
		return 0;

	memset (&dest, 0, sizeof(dest));

	ipvsif_init_dest(&dest, service);

	rc = ipvs_del_dest(&service->virtual_service->ipvs_service, &dest);

	if (rc != 0 && errno != ENOENT) {
		log_printf(LOG_ERR, "IPVS error during RS removal: %s",
		              ipvs_strerror(errno));
		return -1;
	}

#endif
	return 0;
}

int ipvsif_get_service_conns(struct service *service)
{
#ifdef WITH_IPVS
	struct ip_vs_dest_entry *destentry;
	struct ip_vs_get_dests *getdest;
	struct ip_vs_service_entry *svcentry = NULL;
	struct ip_vs_get_services *getsvc;
	int i, conns = -1;
	ipvs_service_t *is = &service->virtual_service->ipvs_service;

	if (!(getsvc = ipvs_get_services())) {
		log_printf(LOG_ERR, "Query of ipvs services failed: %s\n",
				ipvs_strerror(errno));
		return -1;
	}

	for (i = 0; i < getsvc->num_services; i++) {
		svcentry = getsvc->entrytable + i;
		if (svcentry->protocol == is->protocol
			&& svcentry->port == is->port
			&& svcentry->addr == is->addr)
			break;
		else
			svcentry = NULL;
	}

	if (!svcentry) {
		log_printf(LOG_ERR, "Virtual service %s has disappeared",
				service->virtual_service->name);
		return -1;
	}

	if (!(getdest = ipvs_get_dests(svcentry))) {
		log_printf(LOG_ERR,
		    "Could not retrieve destinations for service: "
		    "protocol: %d, port %d: %s",
		    svcentry->protocol,
		    ntohs(svcentry->port),
		    ipvs_strerror(errno));
		return -1;
	}

	for (i = 0; i < getdest->num_dests; i ++) {
		destentry = getdest->entrytable + i;

		if (service->server->address.s_addr == destentry->addr) {
			conns = destentry->activeconns;
			break;
		}
	}

	if (conns == -1) {
		log_printf(LOG_WARN, "Server %s not found in ipvs tables",
		              service->server->name);
	}

	log_printf(LOG_DEBUG, "Service (server %s, protocol %d, port %d) has "
		"%d connections", service->server->name,
		service->virtual_service->ipvs_service.protocol,
	       ntohs(service->port), conns);

	return conns;
#endif
	return 0;
}

void ipvsif_close(void)
{
#ifdef WITH_IPVS
	ipvs_close();
#endif
}
