/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 *
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file feedbackd-master.h
 * Feedbackd server.
 *
 * This program listens for NECP packets from feedbackd client (running on
 * server nodes of load-balancing clusters), and updates ipvs tables according
 * to the data received
 */

#ifndef __HAVE_FEEDBACK_MASTER_H
#define __HAVE_FEEDBACK_MASTER_H 1

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifndef FEEDBACKD_MASTER_LOG
#define FEEDBACKD_MASTER_LOG "/var/log/feedbackd-master.log"
#endif

#ifndef FEEDBACKD_MASTER_CONF
#define FEEDBACKD_MASTER_CONF "/etc/feedbackd-master.xml"
#endif

#include <stdint.h>
#include <netinet/in.h>

#define MAX_SERVICE_NAME_LEN 32

#include "linkedlist.h"
#include "necp.h"
#include "ipvs_interface.h"

#define NIPQUAD(addr) \
        ((unsigned char *)&addr)[0], \
        ((unsigned char *)&addr)[1], \
        ((unsigned char *)&addr)[2], \
        ((unsigned char *)&addr)[3]

/**
 * Server object.
 */
struct server {
	int		sd;		/**< socket descriptor for the server */
	struct in_addr	address;	/**< IP address - this may be
					     referenced after the sd is
					     closed */
	uint8_t		request_id;	/**< Request ID expected */
	uint64_t	seq_sent;	/**< sequence number sent */
	uint64_t	seq_expected;	/**< next sequence number expected */
	time_t		timeout;	/**< keepalive timeout expiry */
	short		retries;	/**< number of keepalive retries sent */
	char		name[16];	/**< name of the server - given by
					     inet_ntoa(address) currently */
};

/**
 * Virtual Service object
 */
struct virtual_service {
	char		name[MAX_SERVICE_NAME_LEN];
	ipvs_service_t	ipvs_service;
};

/**
 * Service object.
 */
struct service {
	struct server	*server;	/**< server this service belongs to */
	struct virtual_service *virtual_service; /**< virtual service */
	uint16_t	port;		/**< port the service is on */
	necp_ftype	forwarding;	/**< how the requests are forwarded */
	uint8_t		current_weight;	/**< current service weight,
					     0 <= weight <= 100 */
	uint8_t		active;		/**< active flag. Inactive servers will
					     be removed when the connection
					     count reaches 0 */
};

#define MAX_NAME_LEN 32

/**
 * List of real servers that we know of
 */
list servers;

/**
 * List of services provided by the real servers
 */
list services;

/**
 * List of virtual services that the director provides
 */
list virtual_services;

/**
 * Main entry point to the program
 * @param argc number of arguments
 * @param argv argument array
 * @return 0 on a sucessful execution, 1 on error
 */
int main(int argc, char **argv);

/**
 * Print a usage message
 * @param progname the name of the program run (argv[0])
 */
void print_usage(const char *progname);
#endif /* HAVE_FEEDBACK_MASTER_H */
