/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file feedbackd-master.c
 */

#include "feedbackd-master.h"

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <assert.h>
#include <string.h>
#include <netdb.h>
#include <errno.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "necp_listener.h"
#include "log.h"
#include "scheduler.h"
#include "ipvs_interface.h"
#include "linkedlist.h"
#include "utils.h"
#include "parser/params.h"

static int debug;
	
const struct option options[] = {
	{"conffile",    1, 0, 'c'},
	{"debug",	0, 0, 'd'},
	{"dry-run",     0, 0, 'n'},
	{"test-config", 0, 0, 't'},
	{"help",        0, 0, 'h'},
	{"usage",       0, 0, 'u'},
	{"version",     0, 0, 'V'},
	{0,             0, 0, 0 }
};


/* temporary vars needed during parsing */
struct virtual_service_conf {
	char *name, *address, *protocol, *port, *scheduler;
};

struct virtual_service_conf *virtual_service_conf;

static int parser_section(char *name)
{
	static struct virtual_service_conf vsc;

	if (virtual_service_conf) {
		log_printf(LOG_ERR, "found a new [virtual-service] section, "
				"but the previous (%s) isn't complete",
				virtual_service_conf->name);
		return 0;
	}

	if (streq(name, "virtual-service")) {
		virtual_service_conf = &vsc;
		memset(virtual_service_conf, 0, sizeof(*virtual_service_conf));
	}

	return 1;
}

static int parser_service_param(struct virtual_service_conf *vsc,
		char *name, char *value)
{
	struct virtual_service *virtual_service;
	struct protoent *proto;
	struct hostent *host;
	int ret, port;

	if (streq(name, "name")) {
		vsc->name = strdup(value);

	} else if (streq(name, "address")) {
		vsc->address = strdup(value);

	} else if (streq(name, "protocol")) {
		vsc->protocol = strdup(value);

	} else if (streq(name, "port")) {
		vsc->port = strdup(value);

	} else if (streq(name, "scheduler")) {
		vsc->scheduler = strdup(value);

	} else {
		log_printf(LOG_ERR, "unknown configuration directive for "
				"a virtual service: '%s'", name);
		return 0;
	}

	/* if we still need parameters for this virtual service, return */
	if (!vsc->name || !vsc->address || !vsc->protocol || !vsc->port ||
			!vsc->scheduler)
		return 1;

	/* we've got everything we need - create a service and parse
	 * the params */
	virtual_service = malloc(sizeof(*virtual_service));
	memset(virtual_service, 0, sizeof(*virtual_service));

	ret = 0;

	strncpy(virtual_service->name, vsc->name,
			sizeof(virtual_service->name) - 1);
	virtual_service->name[sizeof(virtual_service->name) - 1] = '\0';

	proto = getprotobyname(vsc->protocol);
	if (!proto) {
		log_printf(LOG_ERR, "Invalid protocol (%s) given for virtual "
				"service %s", value, virtual_service->name);
		goto out_free;
	}
	virtual_service->ipvs_service.protocol = proto->p_proto;

	/* configure port */
	if (!parse_number(&port, value, 0xffff, NULL)) {
		virtual_service->ipvs_service.port = htons(port & 0xffff);
	} else {
		struct servent *serv;

		/* it's not a number, try service names */
		serv = getservbyname(vsc->port, proto->p_name);

		if (!serv) {
			log_printf(LOG_ERR, "Invalid port (%s) given "
					"for service %s",
					vsc->port, virtual_service->name);
			goto out_free;
		}
		virtual_service->ipvs_service.port = serv->s_port;
	}

	host = gethostbyname(vsc->address);
	if (!host || host->h_addrtype != AF_INET ||
			host->h_length != sizeof(struct in_addr)) {
		log_printf(LOG_ERR, "Could not resolve address %s for virtual "
				"service %s", vsc->address,
				virtual_service->name);
		goto out_free;
	}

	memcpy(&virtual_service->ipvs_service.addr,
			host->h_addr_list[0], sizeof(uint32_t));

	strncpy(virtual_service->ipvs_service.sched_name,
			vsc->scheduler,
			IP_VS_SCHEDNAME_MAXLEN - 1);

	/* add to the virtual services list */
	list_add(&virtual_services, virtual_service);

	/* we're done with the temporary configuration data */
	virtual_service_conf = NULL;

	ret = 1;

out_free:
	free(vsc->name);
	free(vsc->protocol);
	free(vsc->port);
	free(vsc->address);
	free(vsc->scheduler);

	return ret;
}

static int parser_param(char *name, char *value)
{
	int val, ret = 1;

	/* if we're configuring a service, pass the param to it */
	if (virtual_service_conf)
		return parser_service_param(virtual_service_conf, name, value);

	if (streq(name, "keepalive-interval")) {
		if (parse_number(&val, value, INT_MAX, "keepalive-interval"))
			ret = 0;
		else
			scheduler_set_keepalive_interval(val);

	} else if (streq(name, "keepalive-timeout")) {
		if (parse_number(&val, value, INT_MAX, "keepalive-timeout"))
			ret = 0;
		else
			scheduler_set_keepalive_timeout(val);

	} else if (streq(name, "keepalive-retries")) {
		if (parse_number(&val, value, INT_MAX, "keepalive-retries"))
			ret = 0;
		else
			scheduler_set_keepalive_retries(val);

	} else if (streq(name, "removal-interval")) {
		if (parse_number(&val, value, INT_MAX, "removal-interval"))
			ret = 0;
		else
			scheduler_set_removal_interval(val);

	} else if (streq(name, "logfile")) {
		if (!debug && log_setfile(value))
			ret = 0;
	
	} else if (streq(name, "loglevel")) {
		if (!debug && log_setlevel(value))
			ret = 0;

	} else {
		log_printf(LOG_ERR, "Invalid configuration directive %s", name);
		ret = 0;
	}

	return 1;
}

static int configure(char *configfile, int debug)
{
	assert(configfile && strlen(configfile));

	if (debug) {
		log_setlevel("DEBUG");
		log_setfile("-");
	}

	if (!pm_process(configfile, parser_section, parser_param))
		return -1;

	if (list_empty(virtual_services)) {
		log_printf(LOG_ERR, "No virtual services configured");
		return -1;
	}

	return 0;

}

void dump_config(void)
{
	log_printf(LOG_DEBUG, "-- Configuration dump - virtual services:");
	listitem *i;

	list_for_each(virtual_services, i) {
		struct virtual_service *service =
			(struct virtual_service *)i->item;
		log_printf(LOG_DEBUG,
				"Service %s: ("
				"addr: %d.%d.%d.%d, protocol: %d, "
				"port: %d, sched: %s)",
				service->name,
				NIPQUAD(service->ipvs_service.addr),
				service->ipvs_service.protocol,
				ntohs(service->ipvs_service.port),
				service->ipvs_service.sched_name);
	}
}


void print_usage(const char *progname)
{
	fprintf(stderr,
"Usage: %s [-c|--conffile FILE] [-n|--dryrun]\n"
"           [-d|--dont-fork] [-V|--version]\n"
"Options:\n"
"    --conffile FILE:  use FILE to source configuration data\n"
"                      (default: %s)\n"
"\n"
"    --dryrun:         don't change ipvs tables (ipvs tables are still used \n"
"                      to check the number of current connections).\n"
"\n"
"    --dont-fork:      don't run in the background.\n"
"\n"
"    --version:        print version information and exit\n"
, progname, FEEDBACKD_MASTER_CONF);
}

int main(int argc, char **argv)
{
	char *conffile;
	int c, testconfig = 0;

	conffile = FEEDBACKD_MASTER_CONF;
	
	while ((c = getopt_long(argc, argv, "c:dntV?hu", options, NULL))
	       != EOF) {
		switch(c) {
		case 'c':
			conffile = optarg;
			break;
		case 'd':
			debug = 1;
			break;
		case 't':
			testconfig = 1;
			break;
		case 'n':
			ipvsif_set_dryrun(-1);
			break;
		case 'V':
			printf("%s %s\n", PACKAGE_NAME, PACKAGE_VERSION);
			return 0;
		case '?':
		case 'h':
		case 'u':
		default:
			print_usage(argv[0]);
			return -1;
		}
	}

	log_printf(LOG_INFO, "%s starting", argv[0]);
	log_printf(LOG_INFO, "Using config file %s", conffile);

	list_init(&servers);
	list_init(&services);
	list_init(&virtual_services);

	if (configure(conffile, debug)) {
		log_printf(LOG_CRIT, "Configuration failed. Exiting");
		return -1;
	}

	if (testconfig) {
		dump_config();
		log_printf(LOG_VDEBUG, "Configuration successful");
		return 0;
	}

	dump_config();

	/* start necp_listener */
	necp_listener_running = 1;

	if (!debug && fork())
		return 0;

	if (ipvsif_init()) {
		log_printf(LOG_CRIT, "IPVS initialisation failed. Exiting");
		return -1;
	}

	if (ipvsif_init_virtual_services()) {
		log_printf(LOG_CRIT, "IPVS virtual server init failed. "
				"Exiting");
		return -1;
	}

	necp_listen();

	log_stop();
	ipvsif_close();


	return 0;
}
