/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file feedbackd-agent.h
 *
 * Feedbackd agent. Connects to a master (NECP server) and reports load when
 * requested.
 */

#ifndef __HAVE_FEEDBACKD_AGENT_H
#define __HAVE_FEEDBACKD_AGENT_H 1

#include <stdint.h>
#include <stdio.h>

#include "linkedlist.h"
#include "necp.h"

/**
 * Max size of service names + 1
 */
#define MAX_SERVICE_NAME_LEN 256

int running;		/**< Flag to indicate that the client is running */
int stop_service;	/**< Flag to indicate that services should be stopped */

/**
 * Structure to hold information about a service
 */
struct service {
	char     	name[MAX_SERVICE_NAME_LEN]; /**< name of the service */
	uint8_t	 	protocol;	/**< protocol identifier */
	uint16_t 	port;		/**< port number in host byte order */
	uint8_t  	active;		/**< active flag */
	necp_ftype	forwarding;	/**< forwarding type for this service */
	struct monitor  *monitor;	/**< a struct containing monitor
					     config */
};


/**
 * Array of services on this host
 */
list services;

/**
 * Reads the configuration file and sets necessary variables
 * 
 * @param configfile the name of the configuration file
 * @return 0 if the configuration was sucessful, non-zero otherwise
 */
int read_config(const char *configfile);

/**
 * Signal-catching function, called on SIGINT. Stops the client.
 * 
 * @param signum The signal number
 */
void interrupt(int signum);

/**
 * Main select() loop.
 *
 * @return 0 if the exit is graceful (ie, caused by a SIGINT),
 *         non-zero otherwise
 */
int run(void);

/**
 * Get the forwarding id given a string
 * 
 * @param forwarding The string representation of the forwarding type
 *                   (eg "NAT")
 * @return The forwarding id
 */
necp_ftype get_forwarding(const char *forwarding);

/**
 * Print a dump (to the log) of the current configuration
 *
 * pre: services array and nservices have been initialised
 */
void dump_config(void);

/**
 * Print a usage message
 * @param progname the name of the program run (argv[0])
 */
void print_usage(const char *progname);
#endif /* __HAVE_FEEDBACKD_AGENT_H */
