/*
 * feedbackd - dynamic feedback system for LVS
 * Copyright (C) 2002 Jeremy Kerr
 * 
 * This file is part of feedbackd.
 *
 *  feedbackd is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  feedbackd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with feedbackd; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file feedbackd-agent.c
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <dlfcn.h>

#ifdef HAVE_GETOPT_H
#ifdef HAVE_LIBGNUGETOPT
#undef __GNU_LIBRARY__
#endif
#include <getopt.h>
#endif

#include <limits.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>

#include <sys/select.h>
#include <sys/time.h>

#include <signal.h>
#include <errno.h>

#include "necp.h"
#include "necp_client.h"
#include "feedbackd-agent.h"
#include "log.h"
#include "parser/params.h"
#include "utils.h"
#include "plugins/plugin.h"

/**
 * Where to find plugin modules
 */
#ifndef MODULE_DIR
#define MODULE_DIR "/usr/lib/feedbackd-agent/"
#endif

/**
 * Default location of config file
 */
#ifndef FEEDBACKD_AGENT_CONF
#define FEEDBACKD_AGENT_CONF "/etc/feedbackd-agent.xml"
#endif

/**
 * Default log location
 */
#ifndef FEEDBACKD_AGENT_LOG
#define FEEDBACKD_AGENT_LOG "/var/log/feedbackd-agent.log"
#endif

/**
 * How many times to attempt reconnection. 0 = try forever
 */
#ifndef CONNECTION_RETRIES
#define CONNECTION_RETRIES 10
#endif

/**
 * Amount of time (since last data received) to consider connection
 * to master broken
 */
#ifndef MASTER_TIMEOUT
#define MASTER_TIMEOUT 30
#endif

struct in_addr server_address;
static char *moduledir;
static unsigned int connection_retries;
static int master_timeout;
static int debug;

const struct option options[] = {
	{"conffile",    1, 0, 'c'},
	{"test-config", 1, 0, 't'},
	{"debug",	0, 0, 'd'},
	{"help",        0, 0, 'h'},
	{"usage",       0, 0, 'u'},
	{"version",     0, 0, 'V'},
	{0,             0, 0, 0 }
};

struct service_conf {
	char *name, *protocol, *port, *forwarding;
	struct service *service;
};

necp_ftype parse_forwarding(const char *forwarding)
{
	if (!strncasecmp(forwarding, "DR", 2)) {
		return NECP_FT_DROUTE;
	} else if (!strncasecmp(forwarding, "TUN", 3)) {
		return NECP_FT_TUNNEL;
	} else if (!strncasecmp(forwarding, "NAT", 3)) {
		return NECP_FT_MASQ;
	}

	return 0;
}

static int load_service_monitor(struct service *service,
		const char *monitor_obj)
{
	const char *error;
	void *handle;
	char *modpath;
	void *(*create_sym)(struct service *);

	modpath = malloc(strlen(moduledir) + strlen(monitor_obj) + 2);
	if (!modpath) {
		perror("malloc");
		return 0;
	}
	strcpy(modpath, moduledir);
	strcat(modpath, "/");
	strcat(modpath, monitor_obj);

	handle = dlopen(modpath, RTLD_LAZY);
	if (!handle) {
		log_printf(LOG_ERR, "Monitor plugin %s (defined in service %s) "
				"could not be loaded: %s", modpath,
				service->name, dlerror());
		free(modpath);
		return 0;
	}
	free(modpath);

	create_sym = dlsym(handle, "create_monitor");
	if ((error = dlerror())) {
		log_printf(LOG_ERR, "No create_monitor() symbol found "
				"in plugin %s", monitor_obj);
		dlclose(handle);
		return 0;
	}

	service->monitor = create_sym(service);
	/*dlclose(handle);*/

	return service->monitor != NULL;
}

static inline int service_conf_complete(struct service_conf *conf)
{
	struct service *service;
	struct protoent *proto = NULL;
	struct servent *serv = NULL;
	int port;

	/* have we already created the service? */
	if (conf->service)
		return 1;

	/* check for necessary options */
	if (!conf->name) {
		log_printf(LOG_ERR, "No name for service");
		return 0;
	}

	if (!conf->protocol) {
		log_printf(LOG_ERR, "No protocol for service '%s'", conf->name);
		return 0;
	}

	if (!conf->port) {
		log_printf(LOG_ERR, "No port for service '%s'", conf->name);
		return 0;
	}

	if (!conf->forwarding) {
		log_printf(LOG_ERR, "No forwarding type for service '%s'",
				conf->name);
		return 0;
	}

	service = malloc(sizeof(*service));
	if (!service) {
		perror("malloc");
		return 0;
	}
	memset(service, 0, sizeof(*service));

	strncpy(service->name, conf->name, sizeof(service->name) - 1);

	/* configure protocol */
	proto = getprotobyname(conf->protocol);
	if (!proto) {
		log_printf(LOG_ERR, "Invalid protocol (%s) "
				"given for service %s",
				conf->protocol, service->name);
		goto out_free;
	}
	service->protocol = proto->p_proto;

	/* configure port */
	if (!parse_number(&port, conf->port, 0xffff, NULL)) {
		service->port = htons(port & 0xffff);
	} else {
		/* it's not a number, try service names */
		serv = getservbyname(conf->port, proto->p_name);

		if (!serv) {
			log_printf(LOG_ERR, "Invalid port (%s) given "
					"for service %s",
					conf->port, service->name);
			goto out_free;
		}
		service->port = serv->s_port;
	}

	/* configure forwarding */
	service->forwarding = parse_forwarding(conf->forwarding);
	if (!service->forwarding) {
		log_printf(LOG_ERR, "Invalid forwarding (%s) given "
				"for service %s",
				conf->forwarding, service->name);
		goto out_free;
	}

	conf->service = service;
	/* add to the services list */
	list_add(&services, service);

	return 1;

out_free:
	free(service);
	return 0;
}


static int parser_service_param(struct service_conf *service_conf,
		char *name, char *value)
{
	/* if we're configuring the monitor, pass params to its conf routine */
	if (service_conf->service && service_conf->service->monitor) {
		return !service_conf->service->monitor->config(
				service_conf->service->monitor, name, value);
	}

	if (streq(name, "name"))
		service_conf->name = strdup(value);

	else if (streq(name, "protocol"))
		service_conf->protocol = strdup(value);

	else if (streq(name, "port"))
		service_conf->port = strdup(value);

	else if (streq(name, "forwarding"))
		service_conf->forwarding = strdup(value);

	else if (streq(name, "monitor")) {
		if (!service_conf_complete(service_conf)) {
			log_printf(LOG_ERR, "Monitor directive found, but "
					"the service hasn't been defined yet");
			return 0;
		}
		/* since the conf is complete, we know that the service pointer
		 * is valid */
		if (!load_service_monitor(service_conf->service, value))
			return 0;

	} else {
		log_printf(LOG_ERR, "Unknown configuration directive %s", name);
		return 0;
	}

	return 1;
}

/* temporaries used during parsing */
static struct service_conf *service_conf;

static int parser_section(char *name)
{
	static struct service_conf sc;

	if (streq(name, "service")) {
		/* make sure any previous service is complete */ 
		if (service_conf) {
			if (!service_conf->service ||
					!service_conf->service->monitor) {
				log_printf(LOG_ERR, "Found a [service] section,"
						" but the previous (for %s) "
						"isn't complete",
						service_conf->name);
				return 0;
			}
			/* free up the strings we copied */
			free(service_conf->name);
			free(service_conf->protocol);
			free(service_conf->port);
			free(service_conf->forwarding);
		} else {
			service_conf = &sc;
		}

		memset(service_conf, 0, sizeof(*service_conf));

	} else {
		log_printf(LOG_ERR, "Unknown section [%s]", name);
		return 0;
	}

	return 1;
}

static int parser_param(char *name, char *value)
{
	int x;

	/* if we're configuring a service, pass the param to it */
	if (service_conf)
		return parser_service_param(service_conf, name, value);

	if (streq(name, "director")) {
		struct hostent *host;

		host = gethostbyname(value);

		if (!host || host->h_addrtype != AF_INET ||
				host->h_length != sizeof(struct in_addr)) {
			log_printf(LOG_ERR, "Could not resolve "
					"director address %s", value);
			return 0;
		}

		memcpy(&server_address, host->h_addr_list[0],
			       sizeof(server_address));

	} else if (streq(name, "connection-retries")) {
		if (parse_number(&x, value, INT_MAX, "connection-retries"))
			return 0;
		connection_retries = x;

	} else if (streq(name, "master-timeout")) {
		if (parse_number(&master_timeout, value, INT_MAX,
					"master-timeout"))
			return 0;
		if (master_timeout <= 0) {
			log_printf(LOG_ERR, "Invalid master-timeout: %s",
					value);
			return 0;
		}

	} else if (streq(name, "moduledir")) {
		moduledir = strdup(value);

	} else if (streq(name, "logfile")) {
		if (debug)
			return 1;
		if (log_setfile(value))
			return 0;
	
	} else if (streq(name, "loglevel")) {
		if (debug)
			return 1;
		if (log_setlevel(value))
			return 0;

	} else {
		log_printf(LOG_ERR, "Invalid configuration directive %s", name);
		return 0;
	}

	return 1;

}

static int configure(char *configfile)
{
	listitem *i;

	assert(configfile && strlen(configfile));

	if (debug) {
		log_setlevel("DEBUG");
		log_setfile("-");
	}

	if (!pm_process(configfile, parser_section, parser_param))
		return -1;

	if (list_empty(services)) {
		log_printf(LOG_ERR, "No services configured");
		return -1;
	}

	if (!service_conf->service || !service_conf->service->monitor) {
		log_printf(LOG_ERR, "Service definition %s isn't complete",
				service_conf->name);
		return -1;
	}

	/* talloc, one day... */
	if (service_conf->name)
		free(service_conf->name);
	if (service_conf->protocol)
		free(service_conf->protocol);
	if (service_conf->port)
		free(service_conf->port);
	if (service_conf->forwarding)
		free(service_conf->forwarding);

	/* initialise the monitor plugins */
	list_for_each(services, i) {
		struct service *service = (struct service *)(i->item);
		if (service->monitor->init(service->monitor)) {
			log_printf(LOG_ERR, "monitor plugin for service %s "
					"failed to initialise", service->name);
			return -1;
		}
	}

	return 0;
}


int main(int argc, char **argv)
{
	char *conffile;
	int c, x, testconfig = 0;

	/* set defaults */
	conffile = FEEDBACKD_AGENT_CONF;
	moduledir = MODULE_DIR;
	connection_retries = 0;
	master_timeout = MASTER_TIMEOUT;

	while ((c = getopt_long(argc, argv, "c:tdVhu?", options, NULL))
	        != EOF) {

		switch(c) {
		case 'c':
			conffile = optarg;
			break;
		case 't':
			testconfig = 1;
			debug = 1;
			break;
		case 'd':
			debug = 1;
			break;
		case 'V':
			printf("%s %s\n", PACKAGE_NAME, PACKAGE_VERSION);
			return 0;
		case '?':
		case 'h':
		case 'u':
		default:
			print_usage(argv[0]);
			return -1;
		}
	}

	log_printf(LOG_INFO, "%s starting", argv[0]);
	log_printf(LOG_INFO, "Using config file %s", conffile);

	/* random numbers requrired for necp request ids */
	srand((unsigned int)getpid());

	list_init(&services);

	/* read configuration */
	if (configure(conffile)) {
		log_printf(LOG_CRIT, "Configuration failed. Exiting");
		return -1;
	}

	dump_config();

	if (testconfig)
		return 0;

	running = 1;

	if (!debug && fork())
		return 0;

	x = run();

	log_stop();

	return x;
}

void dump_config()
{
	struct service *service;
	listitem *i;

	log_printf(LOG_INFO, "-- Configuration dump:");

	log_printf(LOG_INFO, "Director address: %s", inet_ntoa(server_address));
	log_printf(LOG_INFO, "Module Dir: %s", moduledir);
	log_printf(LOG_INFO, "Master Timeout: %d", master_timeout);
	log_printf(LOG_INFO, "Connection Retries: %d", connection_retries);

	list_for_each(services, i) {
		service = (struct service *)i->item;
		log_printf(LOG_INFO, "Service %s: "
		       "protocol: %3d "
		       "port: %5d "
		       "forwarding: %1d",
		       service->name,
		       service->protocol,
		       ntohs(service->port),
		       service->forwarding);
	}

	log_printf(LOG_INFO, "-- End of configuration dump");
}


int run()
{
	int sd;
	unsigned int retries = 0;
	struct sockaddr_in addr;
	fd_set rfdset, xfdset;
	listitem *item;
	struct service *service;

connect:
	if (!running)
		return -1;

	/* create a socket */
	if ((sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("socket");
		if (connection_retries && retries++ == connection_retries) {
			log_printf(LOG_ERR, "Maximum number of connection "
					"attempts (%d) reached, giving up",
					connection_retries);
			return -1;
		}
		/* Check the signal issues here. --ratz */
		sleep(60);
		goto connect;
	}

	/* create address structure from server_address */
	memset ((char *)&addr, 0, sizeof(addr));
	addr.sin_family	= AF_INET;
	addr.sin_port	= htons(NECP_PORT);
	memcpy(&(addr.sin_addr), &server_address, sizeof(server_address));

	/* connect to server */
	if (connect(sd, (struct sockaddr *)&addr, sizeof(addr))) {
		perror("connect");
		close(sd);
		if (connection_retries && retries++ == connection_retries) {
			log_printf(LOG_ERR, "Maximum number of connection "
					"attempts (%d) reached, giving up",
					connection_retries);
			return -1;
		}
		sleep(60);
		goto connect;
	}

	/* we're connected now */
	retries = 0;

	/* send INIT packet */
	necp_init(sd);

	/* catch interrupt signals and call the interrupt() function,
	 * and ignore SIGPIPE errors (writing to a broken socket) */
	if (signal(SIGINT, interrupt) == SIG_ERR ||
	    signal(SIGPIPE, SIG_IGN) == SIG_ERR) {
		perror("signal");
		return -1;
	}

	/* flag to stop the client process, set by a SIGINT */
	stop_service = 0;

	while (running) {
		int n;
		struct timeval timeout = {master_timeout, 0};

		FD_ZERO(&rfdset);
		FD_SET(sd, &rfdset);
		FD_ZERO(&xfdset);
		FD_SET(sd, &xfdset);

		/* wait for data on the socket */
		n = select(sd+1, &rfdset, NULL, &xfdset, &timeout);

		if (n < 0) {
			if (errno != EINTR) {
				perror("select");
				return -1;
			}
			/* check to see if an interrupt has occurred since
			 * the last iteration */
			if (stop_service) {
				log_printf(LOG_INFO, "Caught SIGINT, stopping "
						"services");
				necp_stop(sd);
			}
		} else if (n == 0) {
			log_printf(LOG_WARN, "Master timed out, "
					"attempting reconnection");
			close(sd);
			goto connect;
		} else {
			if (FD_ISSET(sd, &rfdset)) {
				int ret = data_received(sd);
				if (ret == -2) {
					/* recoverable problem - close socket
					 * and reconnect */
					log_printf(LOG_WARN, "Recoverable "
							"error, attempting "
							"reconnection");
					close(sd);
					goto connect;
				} else if (ret == -1) {
					/* crud. */
					return -1;
				}
			}
		}


	}

	/* change the signal handler back to the default */
	signal(SIGINT, SIG_DFL);
	signal(SIGPIPE, SIG_DFL);

	/* clean up monitors */
	for (item = services.head; item; item = item->next) {
		service = (struct service *)item->item;
		if (service->monitor->destroy) {
			log_printf(LOG_DEBUG, "Removing monitor plugin for "
			               "service %s", service->name);
			service->monitor->destroy(service->monitor);
		}
	}

	log_printf(LOG_INFO, "Shutting down");

	return 0;
}

void interrupt(int signum)
{
	if (stop_service && signum == SIGINT)
		running = 0;
	else
		stop_service = 1;
}

void print_usage(const char *progname)
{
	fprintf(stderr,
"Usage: %s [-c|--conffile FILE] [-f|--dont-fork]"
"           [-t|--test-config]\n"
"\n"
"Options:\n"
"\n"
"    --conffile FILE:    Use configuration file FILE\n"
"                         (instead of default: %s)\n"
"\n"
"    --test-config:      Test configuration data, then exit.\n"
"\n"
"    --dont-fork:        don't run in the background. Useful with -f -\n"
"\n"
"    --version:          print version information and exit\n"
, progname, FEEDBACKD_AGENT_CONF);
}
